// Kanban – Drag & Drop + localStorage (vanilla JS + Bootstrap)
(() => {
  const lsKey = 'kanbanBoardV1';
  let board = loadBoard();
  const lists = {
    todo: document.getElementById('listTodo'),
    doing: document.getElementById('listDoing'),
    done: document.getElementById('listDone'),
  };
  const counters = {
    todo: document.getElementById('countTodo'),
    doing: document.getElementById('countDoing'),
    done: document.getElementById('countDone'),
  };

  // Form + tools
  const addTaskForm = document.getElementById('addTaskForm');
  const taskTitleInput = document.getElementById('taskTitle');
  const taskColumnSelect = document.getElementById('taskColumn');
  const resetBtn = document.getElementById('resetBoardBtn');
  const exportBtn = document.getElementById('exportBtn');
  const importInput = document.getElementById('importInput');

  // Initial render
  renderAll();

  // --- Events ---
  addTaskForm.addEventListener('submit', (e) => {
    e.preventDefault();
    const title = taskTitleInput.value.trim();
    if (!title) return;
    addTask(taskColumnSelect.value, title);
    taskTitleInput.value = '';
    taskTitleInput.focus();
  });

  resetBtn.addEventListener('click', () => {
    if (!confirm('Biztosan alaphelyzetbe állítod a táblát?')) return;
    localStorage.removeItem(lsKey);
    board = getDefaultBoard();
    renderAll();
  });

  exportBtn.addEventListener('click', () => {
    const blob = new Blob([JSON.stringify(board, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'kanban-board.json';
    document.body.appendChild(a);
    a.click();
    setTimeout(() => {
      URL.revokeObjectURL(url);
      a.remove();
    }, 0);
  });

  importInput.addEventListener('change', async (e) => {
    const file = e.target.files?.[0];
    if (!file) return;
    try {
      const text = await file.text();
      const data = JSON.parse(text);
      if (!data?.columns) throw new Error('Hibás formátum');
      board = data;
      saveBoard();
      renderAll();
    } catch (err) {
      alert('Import hiba: ' + err.message);
    } finally {
      e.target.value = '';
    }
  });

  // Drag & drop: listák dragover kezelése
  Object.values(lists).forEach(listEl => {
    listEl.addEventListener('dragover', (e) => {
      e.preventDefault();
      const afterEl = getDragAfterElement(listEl, e.clientY);
      const dragging = document.querySelector('.task-card.dragging');
      if (!dragging) return;
      if (afterEl == null) {
        listEl.appendChild(dragging);
      } else {
        listEl.insertBefore(dragging, afterEl);
      }
    });

    listEl.addEventListener('drop', () => {
      updateBoardFromDOM();
    });
  });

  // --- Functions ---
  function addTask(column, title) {
    const t = { id: ++board.lastId, title, createdAt: Date.now() };
    board.columns[column].push(t);
    saveBoard();
    renderColumn(column);
    updateCounters();
  }

  function createTaskEl(task) {
    const el = document.createElement('div');
    el.className = 'task-card';
    el.draggable = true;
    el.dataset.id = task.id;

    const handler = document.createElement('span');
    handler.className = 'text-muted';
    handler.innerHTML = '<i class="bi bi-grip-vertical"></i>';

    const title = document.createElement('span');
    title.className = 'task-title';
    title.textContent = task.title;
    title.contentEditable = true;
    title.spellcheck = false;
    title.addEventListener('keydown', (e) => {
      if (e.key === 'Enter') { e.preventDefault(); title.blur(); }
    });
    title.addEventListener('blur', () => {
      const newTitle = title.textContent.trim();
      if (!newTitle) { title.textContent = task.title; return; }
      task.title = newTitle;
      saveBoard();
    });

    const actions = document.createElement('div');
    actions.className = 'task-actions d-flex align-items-center gap-1';
    const doneBtn = document.createElement('button');
    doneBtn.className = 'btn btn-outline-success btn-sm';
    doneBtn.title = 'Készre';
    doneBtn.innerHTML = '<i class="bi bi-check2"></i>';
    doneBtn.addEventListener('click', () => moveTaskTo(task.id, 'done'));

    const delBtn = document.createElement('button');
    delBtn.className = 'btn btn-outline-danger btn-sm';
    delBtn.title = 'Törlés';
    delBtn.innerHTML = '<i class="bi bi-trash"></i>';
    delBtn.addEventListener('click', () => {
      if (!confirm('Törlöd ezt a feladatot?')) return;
      removeTask(task.id);
    });

    actions.append(doneBtn, delBtn);
    el.append(handler, title, actions);

    // Drag events for card
    el.addEventListener('dragstart', () => {
      el.classList.add('dragging');
    });
    el.addEventListener('dragend', () => {
      el.classList.remove('dragging');
      updateBoardFromDOM();
    });

    return el;
  }

  function renderColumn(column) {
    const listEl = document.getElementById('list' + capitalize(column));
    listEl.innerHTML = '';
    board.columns[column].forEach(task => {
      listEl.appendChild(createTaskEl(task));
    });
  }

  function renderAll() {
    ['todo','doing','done'].forEach(renderColumn);
    updateCounters();
  }

  function updateCounters() {
    counters.todo.textContent = board.columns.todo.length;
    counters.doing.textContent = board.columns.doing.length;
    counters.done.textContent = board.columns.done.length;
  }

  function removeTask(taskId) {
    ['todo','doing','done'].forEach(col => {
      board.columns[col] = board.columns[col].filter(t => t.id !== taskId);
    });
    saveBoard();
    renderAll();
  }

  function moveTaskTo(taskId, targetCol) {
    let found = null;
    for (const col of ['todo','doing','done']) {
      const idx = board.columns[col].findIndex(t => t.id === taskId);
      if (idx !== -1) {
        found = board.columns[col].splice(idx, 1)[0];
        break;
      }
    }
    if (!found) return;
    board.columns[targetCol].push(found);
    saveBoard();
    renderAll();
  }

  function getDragAfterElement(container, y) {
    const els = [...container.querySelectorAll('.task-card:not(.dragging)')];
    let closest = { offset: Number.NEGATIVE_INFINITY, element: null };
    for (const child of els) {
      const box = child.getBoundingClientRect();
      const offset = y - box.top - box.height / 2;
      if (offset < 0 && offset > closest.offset) {
        closest = { offset, element: child };
      }
    }
    return closest.element;
  }

  function updateBoardFromDOM() {
    const newCols = { todo: [], doing: [], done: [] };
    for (const col of ['todo','doing','done']) {
      const listEl = document.getElementById('list' + capitalize(col));
      const ids = [...listEl.querySelectorAll('.task-card')].map(el => +el.dataset.id);
      const map = taskMap();
      ids.forEach(id => {
        const taskEl = listEl.querySelector('.task-card[data-id="' + id + '"] .task-title');
        const obj = map.get(id);
        if (taskEl && obj) obj.title = taskEl.textContent.trim() || obj.title;
        if (obj) newCols[col].push(obj);
      });
    }
    board.columns = newCols;
    saveBoard();
    updateCounters();
  }

  function taskMap() {
    const m = new Map();
    ['todo','doing','done'].forEach(c => board.columns[c].forEach(t => m.set(t.id, t)));
    return m;
  }

  function loadBoard() {
    try {
      const raw = localStorage.getItem(lsKey);
      if (raw) return JSON.parse(raw);
    } catch {}
    const b = getDefaultBoard();
    localStorage.setItem(lsKey, JSON.stringify(b));
    return b;
  }

  function saveBoard() {
    localStorage.setItem(lsKey, JSON.stringify(board));
  }

  function getDefaultBoard() {
    return {
      lastId: 3,
      columns: {
        todo: [ { id: 1, title: 'UI váz finomítása', createdAt: Date.now() } ],
        doing: [ { id: 2, title: 'Drag&Drop beállítás', createdAt: Date.now() } ],
        done: [ { id: 3, title: 'Projekt inicializálás', createdAt: Date.now() } ]
      }
    };
  }

  function capitalize(s){ return s.charAt(0).toUpperCase() + s.slice(1); }
})();
