// Ultraegyszerű Task modul – vanilla JS + localStorage
(() => {
  const lsKey = 'taskModuleV1';
  let tasks = loadTasks();
  let currentFilter = 'all';

  // Elemtárolók
  const taskForm = document.getElementById('taskForm');
  const taskTitle = document.getElementById('taskTitle');
  const taskDate = document.getElementById('taskDate');
  const taskPriority = document.getElementById('taskPriority');
  const searchInput = document.getElementById('searchInput');
  const statusTabs = document.getElementById('statusTabs');
  const taskList = document.getElementById('taskList');

  // Számlálók
  const countAll = document.getElementById('countAll');
  const countTodo = document.getElementById('countTodo');
  const countDoing = document.getElementById('countDoing');
  const countDone = document.getElementById('countDone');

  // Init
  bindEvents();
  renderTasks();

  function bindEvents() {
    taskForm.addEventListener('submit', handleAddTask);
    statusTabs.addEventListener('click', handleFilterClick);
    searchInput.addEventListener('input', renderTasks);
    taskList.addEventListener('click', handleListClick);
    taskList.addEventListener('change', handleListChange);
    taskList.addEventListener('focusout', handleTitleEdit);
    taskList.addEventListener('keydown', e => {
      if (e.target.classList?.contains('task-title') && e.key === 'Enter') {
        e.preventDefault();
        e.target.blur();
      }
    });
  }

  function handleAddTask(e) {
    e.preventDefault();
    const title = taskTitle.value.trim();
    if (!title) return;
    const newTask = {
      id: uid(),
      title,
      due: taskDate.value || null,
      priority: taskPriority.value,
      status: 'todo',
      createdAt: Date.now()
    };
    tasks.push(newTask);
    saveTasks();
    taskForm.reset();
    renderTasks();
    taskTitle.focus();
  }

  function handleFilterClick(e) {
    const btn = e.target.closest('[data-filter]');
    if (!btn) return;
    currentFilter = btn.dataset.filter;
    statusTabs.querySelectorAll('.nav-link').forEach(b => b.classList.remove('active'));
    btn.classList.add('active');
    renderTasks();
  }

  function handleListClick(e) {
    const li = e.target.closest('li[data-id]');
    if (!li) return;
    const id = li.dataset.id;
    const btn = e.target.closest('[data-action]');
    if (!btn) return;
    const action = btn.dataset.action;
    if (action === 'delete') {
      tasks = tasks.filter(t => t.id !== id);
      saveTasks();
      renderTasks();
    } else if (action === 'toggleDone') {
      const t = findTask(id);
      t.status = t.status === 'done' ? 'todo' : 'done';
      saveTasks();
      renderTasks();
    }
  }

  function handleListChange(e) {
    const li = e.target.closest('li[data-id]');
    if (!li) return;
    const id = li.dataset.id;
    if (e.target.matches('[data-action="changeStatus"]')) {
      const t = findTask(id);
      t.status = e.target.value;
      saveTasks();
      renderTasks();
    }
  }

  function handleTitleEdit(e) {
    if (!e.target.classList?.contains('task-title')) return;
    const li = e.target.closest('li[data-id]');
    if (!li) return;
    const id = li.dataset.id;
    const newTitle = e.target.textContent.trim();
    if (!newTitle) { renderTasks(); return; }
    const t = findTask(id);
    t.title = newTitle;
    saveTasks();
    renderTasks();
  }

  function renderTasks() {
    const q = searchInput.value.toLowerCase();
    const visible = tasks.filter(t =>
      (currentFilter === 'all' || t.status === currentFilter) &&
      t.title.toLowerCase().includes(q)
    );
    taskList.innerHTML = visible.map(t => taskItemTemplate(t)).join('');
    updateCounters();
  }

  function taskItemTemplate(t) {
    const done = t.status === 'done';
    const titleClass = `task-title ${done ? 'text-decoration-line-through text-muted' : ''}`;
    return `
<li class="list-group-item d-flex align-items-center gap-2" data-id="${t.id}">
  <button class="btn btn-sm ${done ? 'btn-success' : 'btn-outline-secondary'}" data-action="toggleDone" title="Készre jelöl">
    <i class="bi ${done ? 'bi-check2' : 'bi-square'}"></i>
  </button>
  <div class="flex-grow-1">
    <span class="${titleClass}" contenteditable="true">${escapeHtml(t.title)}</span>
    <div class="task-meta small text-secondary mt-1">
      <i class="bi bi-flag-fill"></i> ${escapeHtml(t.priority)}
      ${t.due ? ` • <i class="bi bi-calendar-event"></i> ${escapeHtml(formatDate(t.due))}` : ''}
    </div>
  </div>
  <select class="form-select form-select-sm w-auto" data-action="changeStatus" aria-label="Státusz">
    <option value="todo" ${t.status === 'todo' ? 'selected' : ''}>Teendő</option>
    <option value="doing" ${t.status === 'doing' ? 'selected' : ''}>Folyamatban</option>
    <option value="done" ${t.status === 'done' ? 'selected' : ''}>Kész</option>
  </select>
  <button class="btn btn-sm btn-outline-danger" data-action="delete" title="Törlés">
    <i class="bi bi-trash"></i>
  </button>
</li>`;
  }

  function updateCounters() {
    countAll.textContent = tasks.length;
    countTodo.textContent = tasks.filter(t => t.status === 'todo').length;
    countDoing.textContent = tasks.filter(t => t.status === 'doing').length;
    countDone.textContent = tasks.filter(t => t.status === 'done').length;
  }

  function uid() { return Math.random().toString(36).slice(2, 9); }
  function loadTasks() { try { return JSON.parse(localStorage.getItem(lsKey)) || []; } catch { return []; } }
  function saveTasks() { localStorage.setItem(lsKey, JSON.stringify(tasks)); }
  function findTask(id) { return tasks.find(t => t.id === id); }
  function escapeHtml(str) {
    return String(str).replaceAll('&','&amp;').replaceAll('<','&lt;')
      .replaceAll('>','&gt;').replaceAll('"','&quot;').replaceAll("'",'&#039;');
  }
  function formatDate(yyyyMmDd) { return yyyyMmDd.replaceAll('-', '.'); }
})();
